package cs2110;

import java.util.*;

public final class EnrollmentIndex {

    /** The "forward" map associating each course to its list of enrolled students.
     *  The list of students in each entry should contain no duplicates and no null entries and
     *  should be ordered in the iterated order of `studentsToCourses.keySet()`. */
    private final Map<Course, List<Student>> coursesToStudents;

    /** The "reverse" map associating each student to the courses in which they are enrolled
     * students. The list of courses in each entry should contain no duplicates and no null
     * entries and should be ordered in the iterated order of `coursesToStudents.keySet()`.*/
    private final Map<Student, List<Course>> studentsToCourses;
    // Use LinkedHashMap/List to preserve first-seen order

    /**
     * Constructs the forward and reverse enrollment indices based on the given `rosters` that
     * associating each course with its list of enrolled students (possibly containing duplicate
     * and/or null values).
     */
    public EnrollmentIndex(Map<Course, List<Student>> rosters){
        coursesToStudents = new LinkedHashMap<>();
        studentsToCourses = new LinkedHashMap<>();

        for (Course c : rosters.keySet()) {
            for (Student s : rosters.get(c)) {
                if (s == null) {
                    continue;
                }
                if(!coursesToStudents.containsKey(c)) {
                    coursesToStudents.put(c, new ArrayList<>());
                }
                if (!studentsToCourses.containsKey(s)) {
                    studentsToCourses.put(s, new ArrayList<>());
                }
                List<Course> schedule = studentsToCourses.get(s);
                List<Student> roster = coursesToStudents.get(c);
                if (!schedule.contains(c)) {
                    schedule.add(c);
                    roster.add(s);
                }
            }
        }
    }

    /** Returns whether the given student `s` is enrolled in the given course `c`. */
    public boolean isEnrolled(Student s, Course c){
        return coursesToStudents.containsKey(c) && coursesToStudents.get(c).contains(s);
    }

    /** Returns the number of courses in which `s` is enrolled. */
    public int courseCountOf(Student s){
        if (!studentsToCourses.containsKey(s)) {
            return 0;
        }
        return studentsToCourses.get(s).size();
    }

    /** Returns the number of students enrolled in course `c`. */
    public int enrollmentOf(Course c){
        if (!coursesToStudents.containsKey(c)) {
            return 0;
        }
        return coursesToStudents.get(c).size();
    }

    /** Returns an iterator over all the students who are enrolled in at least one course, in the
     * order that they first appeared in the rosters. */
    public Iterator<Student> students() {
        return studentsToCourses.keySet().iterator();
    }

    /** Returns an iterator over all the courses in which `s` is enrolled, listed in the order that
     * they appeared in the rosters. */
    public Iterator<Course> coursesOf(Student s){
        if (!studentsToCourses.containsKey(s)) {
            return Collections.emptyIterator();
        }
        return studentsToCourses.get(s).iterator();
    }

    /** Returns an iterator over all the students who are enrolled in the given course `c`. Each
     * student should be yielded at most once, and no null entries should be returned. */
    public Iterator<Student> studentsIn(Course c){
        if (!coursesToStudents.containsKey(c)) {
            return Collections.emptyIterator();
        }
        return coursesToStudents.get(c).iterator();
    }
}
