package cs2110;

import java.util.HashMap;

/**
 * Represents an undirected bipartite using adjacency lists.
 */
public class BipartiteGraph {

    /**
     * Represents a vertex in a graph responsible for tracking its neighbors.
     */
    public static class Vertex {

        /**
         * The label of this vertex
         */
        String label;

        /**
         * A map associating the labels of this vertex's neighbors with their vertex objects.
         */
        HashMap<String, Vertex> neighbors;

        Vertex(String label) {
            this.label = label;
            neighbors = new HashMap<>();
        }

        /**
         * Returns the label associated with this vertex
         */
        public String label() {
            return label;
        }

        /**
         * Returns whether this vertex has a neighbor with the given `neighborLabel`.
         */
        public boolean hasNeighbor(String neighborLabel) {
            return neighbors.containsKey(neighborLabel);
        }

        /**
         * Enables the iteration over this vertex's neighbors.
         */
        public Iterable<Vertex> neighbors() {
            return neighbors.values();
        }
    }

    /**
     * A map associating the labels of the vertices on the left of the bipartition with their Vertex
     * objects. The keys of this and the `rightVertices` map must be distinct.
     */
    HashMap<String, Vertex> leftVertices;

    /**
     * A map associating the labels of the vertices on the right of the bipartition with their
     * Vertex objects. The keys of this and the `leftVertices` map must be distinct.
     */
    HashMap<String, Vertex> rightVertices;

    public BipartiteGraph() {
        leftVertices = new HashMap<>();
        rightVertices = new HashMap<>();
    }

    /**
     * Adds a vertex with the given label to the left bipartition of this graph. Throws an
     * `IllegalArgumentException` if a vertex with this label already exists in either bipartition.
     */
    public void addLeftVertex(String label) {
        if (hasVertex(label)) {
            throw new IllegalArgumentException(
                    "A vertex labeled \"" + label + "\" already exists.");
        }
        leftVertices.put(label, new Vertex(label));
    }

    /**
     * Adds a vertex with the given label to the right bipartition of this graph. Throws an
     * `IllegalArgumentException` if a vertex with this label already exists in either bipartition.
     */
    public void addRightVertex(String label) {
        if (hasVertex(label)) {
            throw new IllegalArgumentException(
                    "A vertex labeled \"" + label + "\" already exists.");
        }
        rightVertices.put(label, new Vertex(label));
    }

    /**
     * Returns whether this graph contains a Vertex object with the given `label`.
     */
    public boolean hasVertex(String label) {
        return leftVertices.containsKey(label) || rightVertices.containsKey(label);
    }

    /**
     * Returns whether this graph contains an edge between the vertices labeled with `label1` and
     * `label2`.
     */
    public boolean hasEdge(String label1, String label2) {
        if (leftVertices.containsKey(label1)) {
            return leftVertices.get(label1).hasNeighbor(label2);
        }
        return rightVertices.containsKey(label1) && rightVertices.get(label1).hasNeighbor(label2);
    }

    /**
     * Adds an edge between the vertices labeled with `label1` and `label2` to this graph. Throws an
     * `IllegalArgumentException` if either endpoint of this edge is not a vertex in this graph, if
     * the graph already contains an edge between these endpoints, or if the vertices are on the
     * same side of the bipartition.
     */
    public void addEdge(String label1, String label2) {
        if (!hasVertex(label1)) {
            throw new IllegalArgumentException(
                    "Graph does not have a vertex labeled \"" + label1 + "\".");
        }
        if (!hasVertex(label2)) {
            throw new IllegalArgumentException(
                    "Graph does not have a vertex labeled \"" + label2 + "\".");
        }
        if (hasEdge(label1, label2)) {
            throw new IllegalArgumentException("Graph already contains an edge between \"" + label1
                    + "\" and \"" + label2 + "\".");
        }

        Vertex v1, v2;
        if (leftVertices.containsKey(label1)) {
            v1 = leftVertices.get(label1);
            if (leftVertices.containsKey(label2)) {
                throw new IllegalArgumentException("Both vertices in left bipartition");
            }
            v2 = rightVertices.get(label2);
        } else {
            v1 = rightVertices.get(label1);
            if (rightVertices.containsKey(label2)) {
                throw new IllegalArgumentException("Both vertices in right bipartition");
            }
            v2 = leftVertices.get(label2);
        }

        v1.neighbors.put(label2, v2);
        v2.neighbors.put(label1, v1);
    }

    /**
     * Enables iteration over the left vertices in this bipartite graph.
     */
    public Iterable<Vertex> leftVertices() {
        return leftVertices.values();
    }

    /**
     * Enables iteration over the right vertices in this bipartite graph.
     */
    public Iterable<Vertex> rightVertices() {
        return rightVertices.values();
    }
}
