package cs2110;

/**
 * A buffer of values of type T with a fixed capacity.
 */
public class RingBuffer<T> {

    /**
     * Backing array storing the values; length is buffer capacity. Elements that do not correspond
     * to contained values are null.
     */
    private final T[] store;

    /**
     * Index of next value that will be returned by `dequeue()`. Must have `0 <= iHead <
     * store.length`.
     */
    private int iHead;

    /**
     * Index of next element that can be written to by `enqueue()`. Must have `0 <= iTail < *
     * store.length`.
     */
    private int iTail;

    /**
     * Number of values currently contained in buffer.
     */
    private int size;

    private void assertInv() {
        assert iHead >= 0;
        assert iHead < store.length;
        assert iTail >= 0;
        assert iTail < store.length;

        for (int i = iTail; i < store.length && i < iHead; i++) {
            assert store[i] == null;
        }
        if (iHead < iTail) {
            for (int i = 0; i < iHead; i++) {
                assert store[i] == null;
            }
        }
    }

    /**
     * Create a new buffer with the specified capacity.
     */
    @SuppressWarnings("unchecked")
    public RingBuffer(int capacity) {
        store = (T[]) new Object[capacity];
        iHead = 0;
        iTail = 0;
        size = 0;
        assertInv();
    }

    /**
     * Returns true if no element can currently be added to the buffer.
     */
    public boolean isFull() {
        return size == store.length;
    }

    /**
     * Returns true if no element can be consumed from the buffer.
     */
    public boolean isEmpty() {
        return size == 0;
    }

    /**
     * Append a value `x` to the buffer. Throws a `RuntimeException` if this buffer is full.
     */
    public void enqueue(T x) {
        if (isFull()) {
            throw new RuntimeException("Cannot enqueue since buffer is full.");
        }

        assert store[iTail] == null;
        store[iTail] = x;
        iTail = (iTail + 1) % store.length;
        size += 1;

        assertInv();
    }

    /**
     * Remove and return the oldest value in the buffer. Throws a `RuntimeException` if this buffer
     * is empty.
     */
    public T dequeue() {
        if (isEmpty()) {
            throw new RuntimeException("Cannot dequeue since buffer is empty");
        }

        T ans = store[iHead];
        store[iHead] = null;
        iHead = (iHead + 1) % store.length;
        size -= 1;

        assertInv();

        return ans;
    }
}
