package cs2110;

import java.util.Iterator;

/**
 * An unbalanced binary search tree that can store any `Comparable` elements
 */
public class BST<T extends Comparable<T>> extends BinaryTree<T> {
    /*
     * Note: The "T extends Comparable<T>" notation in the type parameter (which we call a "type
     * bound") is notation that says that T can only represent types that are *subtypes* of
     * Comparable<T>. In other words, this is what ensures that the elements stored in this BST
     * can be compared (necessary for the order invariant to make sense).
     */

    /**
     * The left subtree of this BST. All elements in `left` must be "<=" `root`. Must be `null` if
     * `root == null` and `!= null` if `root != null`.
     */
    protected BST<T> left;

    /**
     * The right subtree of this BST. All elements in `right` must be ">=" `root`. Must be `null` if
     * `root == null` and `!= null` if `root != null`.
     */
    protected BST<T> right;

    /**
     * Constructs an empty BST.
     */
    public BST() {
        root = null;
        left = null;
        right = null;
    }

    @Override
    protected BinaryTree<T> left() {
        return (left == null || left.root == null) ? null : left;
    }

    @Override
    protected BinaryTree<T> right() {
        return (right == null || right.root == null) ? null : right;
    }

    /**
     * Returns the number of elements stored in this BST. An empty BST stores 0 elements.
     */
    @Override
    public int size() {
        return (root == null) ? 0 : super.size();
    }

    /**
     * Returns the height of this BST. An empty BST has height 0.
     */
    @Override
    public int height() {
        return (root == null) ? -1 : super.height();
    }

    /**
     * Locates and returns a subtree whose root is `elem`, or the leaf child where `elem` would be
     * located if `elem` is not in this BST. Requires that `elem != null`.
     */
    private BST<T> find(T elem) {
        assert elem != null;
        if (root == null || elem.compareTo(root) == 0) {
            return this;
        } else if (elem.compareTo(root) < 0) { // `elem < root`, recurse left
            return left.find(elem);
        } else { // `elem > root`, recurse right
            return right.find(elem);
        }
    }

    /**
     * Returns whether this BST contains `elem`. Requires that `elem != null`.
     */
    public boolean contains(T elem) {
        assert elem != null; // defensive programming
        return find(elem).root != null;
    }

    /**
     * Requires that `elem != null`.
     */
    public void add(T elem) {
        assert elem != null; // defensive programming
        BST<T> loc = find(elem);
        if (loc.root != null) { // `elem` is already in the BST
            if (loc.right.root == null) {
                // `elem` doesn't have a right subtree; make new `elem` its right child.
                loc = loc.right;
            } else {
                // `elem` has a right subtree; make new `elem` the left child of its successor.
                loc = loc.successorDescendant().left;
                /* This will result in a very unbalanced tree when lots of duplicate elements
                 * are added. An alternate (more complicated) approach can correct this. */
            }
        }
        loc.root = elem;
        loc.left = new BST<>();
        loc.right = new BST<>();
    }

    /**
     * Returns the subtree whose root is the successor of `this.root` in an 
     * in-order traversal. Requires that `right.root != null`, so this BST has 
     * a non-empty right subtree.
     */
    private BST<T> successorDescendant() {
        assert right.root != null; // defensive programming
        BST<T> current = right;
        while (current.left.root != null) {
            current = current.left;
        }
        return current;
    }

    /**
     * Removes the given `elem` from this BST. Requires that `elem != null` and `contains(elem) ==
     * true`
     */
    public void remove(T elem) {
        assert contains(elem); // defensive programming
        BST<T> loc = find(elem);
        if (loc.right.root == null) { // `elem` has an empty right subtree
            loc.supplantWith(loc.left); // supplant it with its left subtree
        } else { // `elem` has a non-empty right subtree; replace it with its successor
            BST<T> successor = loc.successorDescendant();
            loc.root = successor.root; // copy up value in successor
            // successor has an empty left subtree
            successor.supplantWith(successor.right); // supplant successor with its right subtree
        }
    }

    /**
     * Replace this tree with the given `other` tree.
     */
    private void supplantWith(BST<T> other) {
        root = other.root;
        left = other.left;
        right = other.right;
    }
}
