package cs2110;

import cs2110.AdjMatrixGraph.AdjMatrixVertex;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * Represents a directed graph with int-weighted edges using an adjacency matrix.
 */
public class AdjMatrixGraph implements Graph<AdjMatrixVertex, WeightedEdge<AdjMatrixVertex>> {

    /**
     * Represents a vertex in this graph.
     */
    public class AdjMatrixVertex implements Vertex<WeightedEdge<AdjMatrixVertex>> {

        /**
         * The label of this vertex
         */
        private final String label;

        /**
         * The index of this vertex in the edges 2D ArrayList
         */
        private final int index;

        public AdjMatrixVertex(String label, int index) {
            this.label = label;
            this.index = index;
        }

        @Override
        public String label() {
            return label;
        }

        @Override
        public int degree() {
            int degree = 0;
            for (WeightedEdge<AdjMatrixVertex> e : edges.get(index)) {
                if (e != null) {
                    degree += 1;
                }
            }
            return degree;
        }

        @Override
        public boolean hasNeighbor(String headLabel) {
            return hasVertex(headLabel)
                    && edges.get(index).get(vertices.get(headLabel).index) != null;
        }

        @Override
        public WeightedEdge<AdjMatrixVertex> edgeTo(String headLabel) {
            assert hasNeighbor(headLabel); // defensive programming
            return edges.get(index).get(vertices.get(headLabel).index);
        }

        @Override
        public Iterable<WeightedEdge<AdjMatrixVertex>> outgoingEdges() {
            ArrayList<WeightedEdge<AdjMatrixVertex>> outEdges = new ArrayList<>();
            for (WeightedEdge<AdjMatrixVertex> e : edges.get(index)) {
                if (e != null) {
                    outEdges.add(e);
                }
            }
            return outEdges;
        }
    }

    /**
     * A map associating the labels of the vertices with their AdjMatrixVertex objects.
     */
    private final HashMap<String, AdjMatrixVertex> vertices;

    /**
     * A 2D ArrayList whose entries are indexed by the vertex indices. The (i,j)'th entry is null if
     * the vertex indexed i does not have an edge to the vertex indexed j; otherwise, it contains a
     * reference to this edge.
     */
    private final ArrayList<ArrayList<WeightedEdge<AdjMatrixVertex>>> edges;

    public AdjMatrixGraph() {
        vertices = new HashMap<>();
        edges = new ArrayList<>();
    }

    @Override
    public int vertexCount() {
        return vertices.size();
    }

    @Override
    public int edgeCount() {
        int count = 0;
        for (AdjMatrixVertex v : vertices()) {
            count += v.degree();
        }
        return count;
    }

    @Override
    public boolean hasVertex(String label) {
        return vertices.containsKey(label);
    }

    @Override
    public AdjMatrixVertex getVertex(String label) {
        assert hasVertex(label); // defensive programming
        return vertices.get(label);
    }

    @Override
    public boolean hasEdge(String tailLabel, String headLabel) {
        return hasVertex(tailLabel)
                && vertices.get(tailLabel).hasNeighbor(headLabel);
    }

    @Override
    public WeightedEdge<AdjMatrixVertex> getEdge(String tailLabel, String headLabel) {
        assert hasEdge(tailLabel, headLabel); // defensive programming
        return vertices.get(tailLabel).edgeTo(headLabel);
    }

    @Override
    public void addVertex(String label) {
        if (hasVertex(label)) {
            throw new IllegalArgumentException("Graph already contains vertex " + label);
        }
        int index = vertices.size();
        vertices.put(label, new AdjMatrixVertex(label, index));
        edges.add(new ArrayList<>());
        for (ArrayList<WeightedEdge<AdjMatrixVertex>> adjacencies : edges) {
            adjacencies.add(null);
            edges.getLast().add(null);
        }
        edges.getLast().removeLast();
    }

    @Override
    public WeightedEdge<AdjMatrixVertex> addEdge(String tailLabel, String headLabel) {
        if (!vertices.containsKey(tailLabel)) {
            throw new IllegalArgumentException("Graph does not have a vertex " + tailLabel);
        }
        AdjMatrixVertex tail = vertices.get(tailLabel);

        if (!vertices.containsKey(headLabel)) {
            throw new IllegalArgumentException("Graph does not have a vertex " + headLabel);
        }
        AdjMatrixVertex head = vertices.get(headLabel);

        if (tail.hasNeighbor(headLabel)) {
            throw new IllegalArgumentException("Graph already has edge from " + tailLabel + " to "
                    + headLabel);
        }

        WeightedEdge<AdjMatrixVertex> newEdge = new WeightedEdge<>(tail, head);
        edges.get(tail.index).set(head.index, newEdge);
        return newEdge;
    }

    @Override
    public Iterable<AdjMatrixVertex> vertices() {
        return vertices.values();
    }

    @Override
    public Iterable<WeightedEdge<AdjMatrixVertex>> edges() {
        ArrayList<WeightedEdge<AdjMatrixVertex>> edges = new ArrayList<>();
        for (AdjMatrixVertex v : vertices()) {
            for (WeightedEdge<AdjMatrixVertex> e : v.outgoingEdges()) {
                edges.add(e);
            }
        }
        return edges;
    }

    @Override
    public String toString() {
        StringBuilder output = new StringBuilder();
        output.append("   ");
        for (String v : vertices.keySet()) {
            output.append(v + "  ");
        }
        output.append("\n");
        int i = 0;
        for (String v : vertices.keySet()) {
            output.append(v + " [");
            for (int j = 0; j < vertexCount(); j++) {
                if (edges.get(i).get(j) == null) {
                    output.append("   ");
                } else {
                    String label = "" + edges.get(i).get(j).weight();
                    output.append(label + " ".repeat(3 - label.length()));
                }
            }
            output.append("]\n");
            i += 1;
        }
        return output.toString();
    }
}
