package cs2110;

import java.util.ArrayList;
import java.util.HashMap;

/**
 * Represents a directed graph using an adjacency list.
 */
public class AdjListGraph<E extends Edge<AdjListVertex<E>>> implements Graph<AdjListVertex<E>,E> {

    /**
     * A map associating the labels of the vertices with their AdjListVertex objects.
     */
    HashMap<String, AdjListVertex<E>> vertices;

    /**
     * Constructs a new graph initially containing no vertices.
     */
    public AdjListGraph() {
        vertices = new HashMap<>();
    }

    @Override
    public int vertexCount() {
        return vertices.size();
    }

    @Override
    public int edgeCount() {
        int count = 0;
        for (AdjListVertex<E> v : vertices()) {
            count += v.degree();
        }
        return count;
    }

    @Override
    public boolean hasVertex(String label) {
        return vertices.containsKey(label);
    }

    @Override
    public AdjListVertex<E> getVertex(String label) {
        assert hasVertex(label); // defensive programming
        return vertices.get(label);
    }

    @Override
    public boolean hasEdge(String tailLabel, String headLabel) {
        return vertices.containsKey(tailLabel) && vertices.get(tailLabel).hasNeighbor(headLabel);
    }

    @Override
    public E getEdge(String tailLabel, String headLabel) {
        assert hasEdge(tailLabel, headLabel); // defensive programming
        return vertices.get(tailLabel).edgeTo(headLabel);
    }

    @Override
    public void addVertex(String label) {
        if (hasVertex(label)) {
            throw new IllegalArgumentException("Graph already contains vertex " + label);
        }
        vertices.put(label, new AdjListVertex<>(label));
    }

    @Override
    public void addEdge(E edge) {
        AdjListVertex<E> tail = edge.tail();
        AdjListVertex<E> head = edge.head();

        assert vertices.containsKey(tail.label);
        assert vertices.containsKey(head.label);
        assert !this.hasEdge(tail.label, head.label);

        tail.outEdges.put(head.label, edge);
    }

    @Override
    public Iterable<AdjListVertex<E>> vertices() {
        return vertices.values();
    }

    @Override
    public Iterable<E> edges() {
        ArrayList<E> edges = new ArrayList<>();
        for (AdjListVertex<E> v : vertices()) {
            for (E e : v.outgoingEdges()) {
                edges.add(e);
            }
        }
        return edges;
    }

    @Override
    public String toString() {
        StringBuilder output = new StringBuilder();
        for (AdjListVertex<E> v : vertices()) {
            output.append(v.label).append("-> [");
            for (E e : v.outgoingEdges()) {
                output.append(e.head().label()).append(",");
            }
            output.append("\b]\n");
        }
        return output.toString();
    }
}
