package cs2110;

/**
 * Models a checking account in our personal finance app.
 */
public class CheckingAccount implements Account {

    /**
     * The name of this account.
     */
    private String name;

    /**
     * The current balance, in cents, of this account.
     */
    private int balance;

    /**
     * The current month's transaction transactions.
     */
    private StringBuilder transactions;

    /**
     * Constructs a checking account with the given `name` and initial `balance`.
     */
    public CheckingAccount(String name, int balance) {
        this.name = name;
        this.balance = balance;
        this.resetTransactionLog();
    }

    /**
     * Reassigns the transaction log to a new `StringBuilder` object that contains this month's
     * initial account balance
     */
    private void resetTransactionLog() {
        this.transactions = new StringBuilder("Initial Balance: ");
        this.transactions.append(centsToString(this.balance));
        this.transactions.append("\n"); // newline
    }

    /**
     * Converts the given number of `cents` into a String in "$X.XX" format.
     */
    private static String centsToString(int cents) {
        int dollars = cents / 100;
        cents = cents % 100;
        return ("$" + dollars + "." + (cents < 10 ? "0" : "") + cents);
    }

    @Override
    public String name() {
        return this.name;
    }

    @Override
    public int balance() {
        return this.balance;
    }

    /**
     * Deposits the specified `amount`, in cents, to the balance of the account, logs this
     * transaction with the given `memo`, and returns `true` to indicate that this deposit was
     * successful. Requires that `amount > 0`.
     */
    @Override
    public boolean depositFunds(int amount, String memo) {
        assert amount > 0; // defensive programming
        this.balance += amount;
        this.transactions.append(" - Deposit ");
        this.transactions.append(centsToString(amount));
        this.transactions.append(": ");
        this.transactions.append(memo);
        this.transactions.append("\n");
        return true; // we can always add funds to a checking account
    }

    @Override
    public boolean transferFunds(Account receivingAccount, int amount) {
        // Note: This method leverages the parameter subtype substitution rule to achieve subtype
        // polymorphism. We don't want to have to write a separate transfer method to each different
        // type of account.

        assert amount > 0; // defensive programming
        if (amount > this.balance) {
            return false; // insufficient funds
        }
        if (!receivingAccount.depositFunds(amount, "Transfer from " + this.name)) {
            return false; // could not add funds
        }
        this.balance -= amount;
        this.transactions.append(" - Transfer ");
        this.transactions.append(centsToString(amount));
        this.transactions.append(" to ");
        this.transactions.append(receivingAccount.name());
        this.transactions.append("\n");
        return true;
    }

    @Override
    public String transactionReport() {
        this.transactions.append("Final Balance: ");
        this.transactions.append(centsToString(this.balance));
        this.transactions.append("\n");

        String report = this.transactions.toString();
        this.resetTransactionLog();
        return report;
    }
}